
/* Copyright (C) 2001-2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_sbit.c */


#include "fs_itype.h"

#ifdef FS_EMBEDDED_BITMAP

#include "fs_bitmap.h"

/* NOTE: we use the BOLD and OBLIQUE extensions only in STIK fonts, where we */
/* may need embedded bitmaps for Normal, Bold, Oblique, and Bold-Oblique: they use */
/* currently 'undefined' bits of the <flags> field of the BITMAP_SIZE_TABLE */

/***
*** NOTE: we are using bitDepth > 1. (a restricted value) Specifically
*** bitDepth==4, is being used for 4 bit per pixel grayscale images
***
*** at this time composite formats are not supported for grayscale
***
*** if necessary, other formats (PNG, TIFF, GIF, JPG, etc..)
*** can be added in the new "Icon" table.
***
***/

typedef struct
{
    FS_ULONG    data_size;
    FS_ULONG    data_offset;
    FS_USHORT   width;
    FS_USHORT   height;
    FS_SHORT    lo_x;
    FS_SHORT    hi_y;
    FS_USHORT   dx;
    FS_USHORT   dy;
    FS_BYTE     expect_metrics;
    FS_BYTE     byte_padded;
    FS_SHORT    xppm;
    FS_SHORT    yppm;
    FS_SHORT    bpp;        /* bits per pixel */
    FS_USHORT   gIndex;        /* the actual glyphIndex into font */
    FS_USHORT   gPos;        /* the position of gIndex in the INDEX_SUB_TABLE */
    FS_VOID     *glyph_data;
} GRUNT;


#define FLAG_HORIZONTAL  0x01
#define FLAG_VERTICAL    0x02

/* non-standard flags */
#define FLAG_BOLD       0x04    /* stroke_pct >= 5% OR bold_pct >= 3% */
#define FLAG_OBLIQUE    0x08    /* transformation is an oblique of 12 degrees */


/* need these to determine an oblique transform -- embedded's are at 12 degrees */
#define TAN_10_DEGREES 11556
#define TAN_14_DEGREES 16340


typedef struct
{
    FS_USHORT   indexFormat;
    FS_USHORT   imageFormat;
    FS_ULONG    imageDataOffset;
} INDEX_SUB_HEADER;

typedef struct
{
    FS_BYTE    height;
    FS_BYTE    width;
    FS_TINY    bearingX;
    FS_TINY    bearingY;
    FS_BYTE    advance;
} SMALL_GLYPH_METRICS;

/* note: ARM processor pads to 32 bit boundaries so set correct "sizeof(...)" */
#define SIZEOF_SMALL_GLYPH_METRICS 5

typedef struct
{
    FS_BYTE    height;
    FS_BYTE    width;
    FS_TINY    horiBearingX;
    FS_TINY    horiBearingY;
    FS_BYTE    horiAdvance;
    FS_TINY    vertBearingX;
    FS_TINY    vertBearingY;
    FS_BYTE    vertAdvance;
} BIG_GLYPH_METRICS;

typedef struct
{
    FS_USHORT glyphCode;
    FS_USHORT offset;
} CODE_OFFSET_PAIR;

typedef struct
{
    FS_USHORT  glyphCode;
    FS_TINY    xOffset;
    FS_TINY    yOffset;
} EBDT_COMP;

#if (NEED_TO_SWAP)
#define SWAP_EBLC_BST(p)    swap_EBLC_bst(p)
#define SWAP_EBLC_IST(p)    swap_EBLC_ist(p)
#else
#define SWAP_EBLC_BST(p)
#define SWAP_EBLC_IST(p)
#endif
/* local prototypes */
static FS_BITMAP *get_sbitmap(_DS_ TTF *ttf, FS_SHORT xppm, FS_SHORT yppm, FS_USHORT gIndex, FS_SHORT bpp);
static FS_BITMAP *make_sbitmap(_DS_ GRUNT *grunt);
static FS_BITMAP *process_image(_DS_ TTF *ttf, FS_SHORT imageFormat, GRUNT *grunt);
static FS_VOID process_format(_DS_ TTF *ttf, FS_ULONG offset, INDEX_SUB_HEADER *ish, GRUNT *grunt);
static FS_BITMAP *make_comp(_DS_ TTF *ttf, FS_USHORT num, EBDT_COMP *comps, GRUNT *grunt);
static FS_BITMAP *merge_bitmap(_DS_ FS_BITMAP *dst, FS_BITMAP*src, FS_SHORT dx, FS_SHORT dy);
static FS_BITMAP *get_embedded_thing(_DS_ SFNT *sfnt, FS_USHORT gIndex, FS_SHORT bpp);
static FS_VOID swap_EBLC_bst(BITMAP_SIZE_TABLE *p);
static FS_VOID swap_EBLC_ist(INDEX_SUB_TABLE *p);


static FS_VOID set_big_metrics(int vert, GRUNT *grunt, BIG_GLYPH_METRICS *big)
{
    if (vert)
    {
        grunt->lo_x = big->vertBearingX;
        grunt->hi_y = big->vertBearingY - 1;
        grunt->dy = big->vertAdvance;
    }
    else
    {
        grunt->lo_x = big->horiBearingX;
        grunt->hi_y = big->horiBearingY - 1;
        grunt->dx = big->horiAdvance;
    }
}

/****************************************************************/
/*** nb: these assumes that all character objects have the same ***/
/*** structure -- except for the actual <bits> array **************/
/*** this same assumption is made elsewhere ***********************/
/****************************************************************/
FS_BITMAP *get_embedded_bitmap(_DS_ SFNT *sfnt, FS_USHORT gindex)
{
    /* 1 bit per pixel bitmaps */
    return (FS_BITMAP *) get_embedded_thing(_PS_ sfnt, gindex, 1);
}
/****************************************************************/
FS_GRAYMAP *get_embedded_graymap(_DS_ SFNT *sfnt, FS_USHORT gindex, FS_USHORT type)
{
    FS_SHORT bitsPerPixel = 0;
    /* 2,4 or 8 bit per pixel graymaps */
    if ((type & FS_MAP_GRAYMAP4) || (type & FS_MAP_EDGE_GRAYMAP4))
        bitsPerPixel = 4;
    else if (type & FS_MAP_GRAYMAP8 || (type & FS_MAP_EDGE_GRAYMAP8))
        bitsPerPixel = 8;
    else if ((type & FS_MAP_GRAYMAP2) || (type & FS_MAP_EDGE_GRAYMAP2))
        bitsPerPixel = 2;
    return (FS_GRAYMAP *) get_embedded_thing(_PS_ sfnt, gindex, bitsPerPixel);

}

/****************************************************************/
static FS_BITMAP *get_embedded_thing(_DS_ SFNT *sfnt, FS_USHORT gIndex, FS_SHORT bpp)
{
    TTF *ttf;
    SENV *senv;
    FS_BITMAP *bmap = 0;
    int ok = 0;
    FS_FIXED bold_pct;

#ifdef FS_PSEUDO_BOLD
    bold_pct = STATE.bold_pct + STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct;
#else
    bold_pct = 0;
#endif

    /* ? could this happen */
    if (!sfnt || !(LFNT *)(sfnt->lfnt) || !(FS_VOID *)((LFNT *)(sfnt->lfnt)->fnt) || !(SENV *)(sfnt->senv) )
        return 0;

    ttf = (FS_VOID *)((LFNT *)(sfnt->lfnt)->fnt);
    senv = (SENV *)(sfnt->senv);

    /* ? no embedded bitmaps */
    if (!ttf->eblc_offset)
        return 0;

    /*** stik fonts can handle bold and simple oblique ***/
    if (sfnt->lfnt->fontflags & FONTFLAG_STIK)
        ok = sfnt->user_scale[0] > 0 && sfnt->user_scale[2] == 0 && sfnt->user_scale[3] > 0;
    else /* outlines can not */
        ok = senv->vanilla &&
             ( ( (STATE.flags & FLAGS_ADD_WEIGHT) && (STATE.lpm <= 26) ) ||
               (!(STATE.flags & FLAGS_ADD_WEIGHT) && (bold_pct == 0)   ) );

    if (ok)
        bmap = get_sbitmap(_PS_ ttf, senv->xppm, senv->yppm, gIndex, bpp);

    if (bmap)
    {
#ifdef FS_STATS
        embedded_bmap++;
#endif /* FS_STATS */
        bmap->embedded = true;
        bmap->bitsPerPixel = bpp;
        if (bpp == 1)
            bmap->type = FS_MAP_BITMAP;
        else if (bpp == 4)
            bmap->type = FS_MAP_GRAYMAP4;
        else if (bpp == 2)
            bmap->type = FS_MAP_GRAYMAP2;
        else if (bpp == 8)
            bmap->type = FS_MAP_GRAYMAP8;
    }

    return bmap;
}

/****************************************************************/
static FS_VOID swap_EBLC_bst(BITMAP_SIZE_TABLE *p)
{
    p->indexSubTableArrayOffset = SWAPL(p->indexSubTableArrayOffset);
    p->indexTableSize = SWAPL(p->indexTableSize);
    p->numberOfIndexSubTables = SWAPL(p->numberOfIndexSubTables);
    p->colorRef = SWAPL(p->colorRef);
    p->startGlyphIndex = SWAPW(p->startGlyphIndex);
    p->endGlyphIndex = SWAPW(p->endGlyphIndex);
}

static FS_VOID swap_EBLC_ist(INDEX_SUB_TABLE *p)
{
    p->firstGlyphIndex = SWAPW(p->firstGlyphIndex);
    p->lastGlyphIndex = SWAPW(p->lastGlyphIndex);
    p->additionalOffsetToIndexSubTable = SWAPL(p->additionalOffsetToIndexSubTable);
}

TTF_EBLC_PARTIAL *get_EBLC_partial(_DS_ TTF *ttf)
{
    FS_ULONG offset, size, i, j;
    TTF_EBLC_PARTIAL *peblc;
    BITMAP_SIZE_TABLE_ARRAY *pbsta;
    BITMAP_SIZE_TABLE *pbst = NULL;
    INDEX_SUB_TABLE *pist = NULL;

    /* First, read in the number of "bitmapSizeTable" entries */
    size = sizeof(TTF_EBLC_PARTIAL);
    offset = ttf->eblc_offset;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "TTF_EBLC_PARTIAL";
#endif
    peblc = (TTF_EBLC_PARTIAL *)ttf_read(_PS_ ttf, offset, size);
    if ((peblc == NULL) || STATE.error)
    {
        if (peblc)
            FSS_free(_PS_ peblc);
        return 0;
    }

    peblc->version = SWAPL(peblc->version);
    peblc->numSizes = SWAPL(peblc->numSizes);

    /* Next, read in all "bitmapSizeTable" entries */
    size = peblc->numSizes * sizeof(BITMAP_SIZE_TABLE_ARRAY);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "BITMAP_SIZE_TABLE_ARRAY";
#endif
    peblc->bsta = pbsta = (BITMAP_SIZE_TABLE_ARRAY *)FSS_calloc(_PS_ size);
    if (pbsta == NULL)  /* Coverity has difficulty with "if (STATE.error)" */
    {
        FSS_free(_PS_ peblc);
        return 0;
    }

    for (i = 0; i < peblc->numSizes; i++)
    {
        size = sizeof(BITMAP_SIZE_TABLE);
        offset = ttf->eblc_offset + sizeof(FS_FIXED) + sizeof(FS_ULONG) +
                 (i * size);
        ttf_read_buf(_PS_ ttf, offset, size, (FS_BYTE *)&pbsta[i].bst);
        if (STATE.error)
        {
            FSS_free(_PS_ pbsta);
            FSS_free(_PS_ peblc);
            return 0;
        }
        SWAP_EBLC_BST(&pbsta[i].bst);
    }

    /* Now read in all "indexSubTable" entries */
    for (i = 0; i < peblc->numSizes; i++)
    {
        FS_ULONG ii;

        pbst = &pbsta[i].bst;
        size = pbst->numberOfIndexSubTables * sizeof(INDEX_SUB_TABLE);
        offset = ttf->eblc_offset + pbst->indexSubTableArrayOffset;
#ifdef FS_MEM_DBG
        STATE.memdbgid = "INDEX_SUB_TABLE";
#endif
        pbsta[i].ist = (INDEX_SUB_TABLE *)ttf_read(_PS_ ttf, offset, size);
        if ((pbsta[i].ist == NULL) || STATE.error)
        {
            for (j = 0; j < peblc->numSizes; j++)
            {
                if (pbsta[j].ist)
                    FSS_free(_PS_ pbsta[j].ist);
            }
            FSS_free(_PS_ pbsta);
            FSS_free(_PS_ peblc);
            return 0;
        }
        for (ii = 0; ii < pbst->numberOfIndexSubTables; ii++)
        {
            pist = pbsta[i].ist;
            SWAP_EBLC_IST(pist + ii);
        }
    }
    return peblc;
}

/****************************************************************/
static FS_BITMAP *get_sbitmap(_DS_ TTF *ttf, FS_SHORT xppm, FS_SHORT yppm, FS_USHORT gIndex, FS_SHORT bpp)
{
    FS_ULONG i, j;
    GRUNT grunt;
    FS_ULONG numTables, table_offset;
    FS_BITMAP *bmap;
    int want_bold = 0, is_bold = 0;
    int want_oblique = 0, is_oblique = 0;
    BITMAP_SIZE_TABLE *bst;
    TTF_EBLC_PARTIAL *peblc;
    BITMAP_SIZE_TABLE_ARRAY *pbsta;
    peblc = (TTF_EBLC_PARTIAL *)(ttf->ttf_EBLC_partial);
    pbsta = (BITMAP_SIZE_TABLE_ARRAY *)(peblc->bsta);
    numTables = peblc->numSizes;
    table_offset = ttf->eblc_offset;

    /** not only does FS_STIK need to be defined, it needs to be a STIK font **/
#ifdef FS_STIK
    if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
    {
        FS_FIXED *scale = STATE.cur_sfnt->user_scale;

        /* only want upright (but possibly obliqued) characters */
        if (scale[0] > 0 && scale[2] == 0 && scale[3] > 0)
        {
            /* If an oblique angle is specified which is outside the */
            /* valid range (10 - 14 degrees), return immediately.    */
            if ((scale[1] > 0) &&
                ((scale[1] < FixMul(scale[3], TAN_10_DEGREES)) ||
                (scale[1] > FixMul(scale[3], TAN_14_DEGREES))  ) )
                return 0;

            want_bold = STATE.stroke_pct > 5243; /* 8% stroke pct threshold */

            /* upright ?  */
            if (scale[1] == 0)
                want_oblique = 0;
            /* between 10 and 14 degrees of oblique ? */
            else if (FixMul(scale[3], TAN_10_DEGREES) <= scale[1] && scale[1] <= FixMul(scale[3], TAN_14_DEGREES))
                want_oblique = 1;
            /* won't match ANY embedded bitmap ... quit */
            else
                return 0;

        }
    }

#ifdef SBIT_DEBUG
    FS_PRINTF(("%% want_bold=%d want_oblique=%d\n", want_bold, want_oblique));
#endif
#endif /* FS_STIK */

    /* for each table ... */
    for (i = 0; i < numTables; i++)
    {
        bst = &pbsta[i].bst;

#ifdef SBIT_DEBUG
        if (bst->ppmX == xppm && bst->ppmY)
        {
            FS_USHORT z;

            FS_PRINTF(("bst[%d]\n", (int)i));
            FS_PRINTF(("\tindexSubTableArrayOffset=%lu\n", bst->indexSubTableArrayOffset));
            FS_PRINTF(("\tindexTableSize=%lu\n", bst->indexTableSize));
            FS_PRINTF(("\tnumberOfIndexSubTables=%lu\n", bst->numberOfIndexSubTables));
            FS_PRINTF(("\tcolorRef=%lu\n", bst->colorRef));
            z = bst->startGlyphIndex;
            FS_PRINTF(("\tstartGlyphIndex=%d 0x%04x\n", z, z));
            z = bst->endGlyphIndex;
            FS_PRINTF(("\tendGlyphIndex=%d 0x%04x\n", z, z));
            FS_PRINTF(("\tflag =%02x ppmX=%d  ppmY=%d  bitDept=%d\n", bst->flags, bst->ppmX, bst->ppmY, bst->bitDepth));
        }
#endif /* SBIT_DEBUG */

        /* ? correct bold and oblique setting */
        is_bold = (bst->flags & FLAG_BOLD) == FLAG_BOLD;
        if (want_bold != is_bold)
            continue;

        is_oblique = (bst->flags & FLAG_OBLIQUE) == FLAG_OBLIQUE;
        if (want_oblique != is_oblique)
            continue;

        /* ? bad size or depth */
        if (bst->ppmX != xppm || bst->ppmY != yppm || bst->bitDepth != bpp)
            continue;

        /* ? in range */
        if (gIndex >= bst->startGlyphIndex && gIndex <= bst->endGlyphIndex)
        {
            FS_ULONG indexSubTableArrayOffset = bst->indexSubTableArrayOffset;
            FS_ULONG numberOfIndexSubTables = bst->numberOfIndexSubTables;

            INDEX_SUB_TABLE *pista_buff;

            pista_buff = (INDEX_SUB_TABLE *)(pbsta[i].ist);
            for (j = 0; j < numberOfIndexSubTables; j++)
            {
                INDEX_SUB_TABLE *st;

                st = pista_buff + j;

                /* ? probable */
                if (gIndex >= st->firstGlyphIndex && gIndex <= st->lastGlyphIndex)
                {
                    FS_ULONG offset;

                    INDEX_SUB_HEADER ish;
                    offset = table_offset + indexSubTableArrayOffset + st->additionalOffsetToIndexSubTable;

                    /* read indexSubHeader */
                    ttf_read_buf(_PS_ ttf, offset, sizeof(INDEX_SUB_HEADER), (FS_BYTE *)&ish);
                    if (STATE.error)
                    {
                        return 0;
                    }
                    offset += sizeof(INDEX_SUB_HEADER);

#if NEED_TO_SWAP
                    ish.indexFormat = SWAPW(ish.indexFormat);
                    ish.imageFormat = SWAPW(ish.imageFormat);
                    ish.imageDataOffset = SWAPL(ish.imageDataOffset);
#endif

                    /* small glyph format?, check for direction now */
                    if (ish.imageFormat == 1 || ish.imageFormat == 2 || ish.imageFormat == 8)
                    {
                        if ((STATE.flags & FLAGS_VERTICAL_ON) && !(bst->flags & FLAG_VERTICAL))
                        {
                            /* wanted vertical metrics, didn't get them */
                            continue;
                        }
                        if (!(STATE.flags & FLAGS_VERTICAL_ON) && !(bst->flags & FLAG_HORIZONTAL))
                        {
                            /* wanted horizontal metrics, didn't get them */
                            continue;
                        }
                    }
                    SYS_MEMSET(&grunt, 0, sizeof(GRUNT));
                    grunt.xppm = xppm;
                    grunt.yppm = yppm;
                    grunt.bpp = bpp;
                    grunt.gIndex = gIndex;    /* for sparse formats */
                    grunt.gPos = gIndex - st->firstGlyphIndex;    /* for dense formats  */
                    process_format(_PS_ ttf, offset, &ish, &grunt);
                    if (STATE.error)
                    {
                        return 0;
                    }

                    bmap = process_image(_PS_ ttf, ish.imageFormat, &grunt);
                    if (STATE.error)
                        return 0;

                    return bmap;
                }
            }
        }
    }
    /* not in any BST's */
    return 0;
}

/****************************************************************/
static FS_BITMAP *make_sbitmap(_DS_ GRUNT *grunt)
{
    FS_USHORT bpl = (FS_USHORT)0;
    FS_BITMAP *bmap;
    FS_ULONG size;
    FS_SHORT bpp = grunt->bpp;
    int extra = 0;
    int simple;

    if (grunt->glyph_data == 0)
        return 0;

    switch (bpp)
    {
    case 1:        /* bitmaps */
        bpl = (7 + grunt->width) / 8;
        break;
    case 2:
        bpl = (3 + grunt->width) / 4;
        break;
    case 4:        /* graymaps */
        bpl = (1 + grunt->width) / 2;
        break;
    case 8:
        bpl = grunt->width;
        break;
    default:
        break;
    }

    size = offsetof(FS_BITMAP, bits);
    size += bpl * grunt->height + extra;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_BITMAP";
#endif
    bmap = (FS_BITMAP *) FSS_calloc(_PS_ size);
    if (bmap == NULL) /* Coverity does not like "if (STATE.error)" */
        return 0;

    bmap->size = size;
    bmap->width = grunt->width;
    bmap->bpl = bpl;
    bmap->height = grunt->height;
    bmap->lo_x = grunt->lo_x;
    bmap->hi_y = grunt->hi_y;
    bmap->i_dx = grunt->dx;
    bmap->i_dy = grunt->dy;

    /* ? is the data actually byte padded */
    simple = grunt->byte_padded ||            /* they say so */
             (bpp == 1 && !(grunt->width & 7)) ||  /* bitmaps of width==8x */
             (bpp == 4 && !(grunt->width & 1));  /* graymaps of width==2x */

    if (simple)
        SYS_MEMCPY(bmap->bits, (FS_BYTE *)grunt->glyph_data + grunt->data_offset, extra + bpl * grunt->height);
    else
    {
        /* source bits are contiguous around line ends ... this is UGLY */
        FS_BYTE *sp = (FS_BYTE *)grunt->glyph_data + grunt->data_offset;
        FS_ULONG sb = 0;
        FS_BYTE *dp = bmap->bits;
        FS_LONG i, ii = bmap->height;
        FS_LONG j, jj = bmap->width * bpp;

        for (i = 0; i < ii; i++)
        {
            for (j = 0; j < jj; j++)
            {
                if (sp[sb >> 3] & fs_mask[sb & 7])
                    dp[j >> 3] |= fs_mask[j & 7];
                sb++;
            }
            dp += bpl;
        }
    }
    return bmap;
}

/****************************************************************/
static FS_BITMAP *process_image(_DS_ TTF *ttf, FS_SHORT imageFormat, GRUNT *grunt)
{
    FS_ULONG table_offset;
    FS_ULONG table_size;
    FS_BITMAP *bmap;
    int vert = STATE.flags & FLAGS_VERTICAL_ON;

    /* if we did not find the index */
    if (grunt->data_offset == 0 || grunt->data_size == 0)
        return 0;

    /* get the EBDT table offset */
    get_ttf_table_info(_PS_ ttf, TAG_EBDT, &table_offset, &table_size);
    if (STATE.error)
        return 0;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "grunt->glyph_data";
#endif
    grunt->glyph_data = ttf_read(_PS_ ttf, table_offset + grunt->data_offset, grunt->data_size);
    if (STATE.error || (grunt->glyph_data == 0))
        return 0;

    /* now process the image formats */
    switch (imageFormat)
    {
    case 1:
    case 2:
        {
            SMALL_GLYPH_METRICS *small = (SMALL_GLYPH_METRICS *)grunt->glyph_data;

            grunt->width = small->width;
            grunt->height = small->height;
            grunt->lo_x = small->bearingX;
            grunt->hi_y = small->bearingY - 1;
            if (vert)
                grunt->dy = small->advance;
            else
                grunt->dx = small->advance;

            grunt->data_offset = SIZEOF_SMALL_GLYPH_METRICS;
            grunt->byte_padded = (imageFormat == 1);

            /* fix a Coverity tainted variable complaint */
            if (grunt->height > 255)
            {
                STATE.error = ERR_BAD_GLYF_FORMAT;
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
            bmap = make_sbitmap(_PS_ grunt);
            if (STATE.error)
            {
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
        }
        break;

    case 3:
    case 4:
        /* failure condition */
        FSS_free(_PS_ grunt->glyph_data);
        return 0;

    case 5:
        {
            if (grunt->expect_metrics)
            {
                STATE.error = EBLC_NO_METRICS;
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
            grunt->data_offset = 0;
            grunt->byte_padded = 0;
            bmap = make_sbitmap(_PS_ grunt);
            if (STATE.error)
            {
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
        }
        break;

    case 6:
    case 7:
        {
            BIG_GLYPH_METRICS *big = (BIG_GLYPH_METRICS *)grunt->glyph_data;

            grunt->width = big->width;
            grunt->height = big->height;
            set_big_metrics(vert, grunt, big);
            grunt->data_offset = sizeof(BIG_GLYPH_METRICS);
            grunt->byte_padded = (imageFormat == 6);

            /* fix a Coverity tainted variable complaint */
            if (grunt->height > 255)
            {
                STATE.error = ERR_BAD_GLYF_FORMAT;
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
            bmap = make_sbitmap(_PS_ grunt);
            if (STATE.error)
            {
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
        }
        break;

    case 8:
        {
            /* composite, small metrics */
            SMALL_GLYPH_METRICS *small = (SMALL_GLYPH_METRICS *)grunt->glyph_data;
            EBDT_COMP *comps;
            FS_USHORT i, num;
            FS_BYTE *p;
            FS_VOID *ptr;

            grunt->width = small->width;
            grunt->height = small->height;
            grunt->lo_x = small->bearingX;
            grunt->hi_y = small->bearingY - 1;
            if (vert)
                grunt->dy = small->advance;
            else
                grunt->dx = small->advance;

            p = (FS_BYTE *)grunt->glyph_data;
            p += 1 + SIZEOF_SMALL_GLYPH_METRICS;

            ptr = p;
            num = *(FS_USHORT *)ptr;
            ptr = p += 2;
            comps = (EBDT_COMP *)ptr;
#if NEED_TO_SWAP
            num = SWAPW(num);
            for (i = 0; i < num; i++)
                comps[i].glyphCode = SWAPW(comps[i].glyphCode);
#endif
            /* fix a Coverity tainted variable complaint */
            if (((grunt->hi_y + 1) < -128) || ((grunt->hi_y + 1) > 127))
            {
                STATE.error = ERR_BAD_GLYF_FORMAT;
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
            bmap = make_comp(_PS_ ttf, num, comps, grunt);
            if (STATE.error)
            {
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
        }
        break;
    case 9:
        /* composite, big metrics */
        {
            BIG_GLYPH_METRICS *big = (BIG_GLYPH_METRICS *)grunt->glyph_data;
            EBDT_COMP *comps;
            FS_USHORT i, num;
            FS_BYTE *p;
            FS_VOID *ptr;

            grunt->width = big->width;
            grunt->height = big->height;
            set_big_metrics(vert, grunt, big);
            p = (FS_BYTE *)grunt->glyph_data;
            p += sizeof(BIG_GLYPH_METRICS);
            ptr = p;
            num = *(FS_USHORT *)ptr;
            ptr = p += 2;
            comps = (EBDT_COMP *)ptr;
#if NEED_TO_SWAP
            num = SWAPW(num);
            for (i = 0; i < num; i++)
                comps[i].glyphCode = SWAPW(comps[i].glyphCode);
#endif
            bmap = make_comp(_PS_ ttf, num, comps, grunt);
            if (STATE.error)
            {
                FSS_free(_PS_ grunt->glyph_data);
                return 0;
            }
        }
        break;

    default:
        FSS_free(_PS_ grunt->glyph_data);
        return 0;
    }

    /* make_comp or make_sbitmap could fail ... */
    if (bmap == 0)
        return 0;

    FSS_free(_PS_ grunt->glyph_data);

    bmap->dx = ((FS_LONG)bmap->i_dx) << 16;
    bmap->dy = ((FS_LONG)bmap->i_dy) << 16;
    if (grunt->bpp == 1)
    {
        bmap = trim_bitmap(_PS_ bmap);
        if (STATE.error)
        {
            return 0;
        }
    }

    return bmap;
}

/****************************************************************/
static FS_VOID process_format(_DS_ TTF *ttf, FS_ULONG offset, INDEX_SUB_HEADER *ish, GRUNT *grunt)
{
    FS_ULONG l_temp;
    FS_USHORT s_temp;
    BIG_GLYPH_METRICS big;
    int vert = STATE.flags & FLAGS_VERTICAL_ON;

    l_temp = 0;    /* without this line, coverity reports uninit_use */

    big.width = 0;    /* without this line, coverity reports uninit_use */
    big.height = 0;    /* without this line, coverity reports uninit_use */

    /* in case we can not find this index in the subtable */
    grunt->data_offset = 0;

    /* read in proper indexSubTable */
    switch (ish->indexFormat)
    {
    case 1:
        /* variable metrics, FS_LONG offsets */
        grunt->expect_metrics = 1;
        offset += grunt->gPos * 4;
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&l_temp);
        if (STATE.error)
            return;
        grunt->data_offset = SWAPL(l_temp);
        ttf_read_buf(_PS_ ttf, offset + 4, 4, (FS_BYTE *)&l_temp);
        if (STATE.error)
            return;

        grunt->data_size = SWAPL(l_temp);    /* so far just the next offset */
        grunt->data_size -= grunt->data_offset;    /* now the real size */
        grunt->data_offset += ish->imageDataOffset;
        break;
    case 2:
        /* constant size */
        grunt->expect_metrics = 0;
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&l_temp);
        if (STATE.error)
            return;
        grunt->data_size = SWAPL(l_temp);
        grunt->data_offset = ish->imageDataOffset + (grunt->gPos * grunt->data_size);

        ttf_read_buf(_PS_ ttf, offset + 4, sizeof(BIG_GLYPH_METRICS), (FS_BYTE *)&big);
        if (STATE.error)
            return;

        grunt->width = big.width;
        grunt->height = big.height;
        set_big_metrics(vert, grunt, &big);
        break;
    case 3:
        /* variable metrics, FS_SHORT offsets */
        s_temp = 0;    /* without this line, coverity reports uninit_use */
        grunt->expect_metrics = 1;
        offset += grunt->gPos * 2;
        ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&s_temp);
        if (STATE.error)
            return;
        grunt->data_offset = SWAPW(s_temp);
        ttf_read_buf(_PS_ ttf, offset + 2, 2, (FS_BYTE *)&s_temp);
        if (STATE.error)
            return;
        grunt->data_size = SWAPW(s_temp);    /* so far just the next offset */
        grunt->data_size -= grunt->data_offset;    /* now the real size */
        grunt->data_offset += ish->imageDataOffset;
        break;
    case 4:
        /* variable metrics, sparse indices */
    {
        FS_ULONG i, num;
        CODE_OFFSET_PAIR *s;
        FS_USHORT sgIndex = SWAPW(grunt->gIndex);

        grunt->expect_metrics = 1;
        ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&l_temp);
        if (STATE.error)
            return;
        num = SWAPL(l_temp);

#ifdef FS_MEM_DBG
        STATE.memdbgid = "CODE_OFFSET_PAIR";
#endif
        s = (CODE_OFFSET_PAIR *)ttf_read(_PS_ ttf, offset + 4, (1 + num) * sizeof(CODE_OFFSET_PAIR));
        if (!s || STATE.error)
        {
            STATE.error = SUCCESS;
            return;
        }
        for (i = 0; i < num; i++)
        {
            /* rather than swap each incoming glyphCode ... compare against pre-swapped gIndex */
            if (s[i].glyphCode == sgIndex)
            {
                grunt->data_offset = SWAPW(s[i].offset);
                grunt->data_size = SWAPW(s[i + 1].offset) - grunt->data_offset;
                grunt->data_offset += ish->imageDataOffset;
                break;
            }
        }
        FSS_free(_PS_ s);

        /* didn't find the gIndex ... NOT an error */
        if (i == num)
            return;
        break;
    }
    case 5:
        /* constant metrics, sparse codes */
        {
            FS_ULONG i, num;
            FS_USHORT *s;
            FS_USHORT sgIndex = SWAPW(grunt->gIndex);

            grunt->expect_metrics = 0;
            ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&l_temp);
            if (STATE.error)
                return;
            offset += 4;
            grunt->data_size = SWAPL(l_temp);

            ttf_read_buf(_PS_ ttf, offset, sizeof(BIG_GLYPH_METRICS), (FS_BYTE *)&big);
            if (STATE.error)
                return;
            offset += sizeof(BIG_GLYPH_METRICS);
            grunt->width = big.width;
            grunt->height = big.height;
            set_big_metrics(vert, grunt, &big);
            ttf_read_buf(_PS_ ttf, offset, 4, (FS_BYTE *)&l_temp);
            if (STATE.error)
                return;
            offset += 4;
            num = SWAPL(l_temp);

#ifdef FS_MEM_DBG
            STATE.memdbgid = "codes(sparse)";
#endif
            s = (FS_USHORT *) ttf_read(_PS_ ttf, offset, num * 2);
            if (!s || STATE.error)
            {
                STATE.error = SUCCESS;
                return;
            }
            for (i = 0; i < num; i++)
            {
                /* compare incoming against pre swapped grunt->gIndex */
                if (sgIndex == s[i])
                    break;
            }
            FSS_free(_PS_ s);
            if (i < num)
                grunt->data_offset = ish->imageDataOffset + (i * grunt->data_size);
            else
                return;    /* NOT an error */
        }
        break;

    default:
        STATE.error = ERR_INDEX_FORMAT;
        return;
    }
}

/****************************************************************/
static FS_BITMAP *make_comp(_DS_ TTF *ttf, FS_USHORT num, EBDT_COMP *comps, GRUNT *grunt)
{
    FS_USHORT bpl;
    FS_LONG size;
    FS_BITMAP *bmap;
    FS_USHORT i;

    /* composites currently only supported for bpp==1 */
    if (grunt->bpp != 1)
        return 0;

    /* make a bitmap sized for final result */
    bpl = (7 + grunt->width) / 8;
    size = offsetof(FS_BITMAP, bits);
    size += bpl * grunt->height;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_BITMAP";
#endif
    bmap = (FS_BITMAP *) FSS_calloc(_PS_ size);
    if (bmap == NULL) /* Coverity does not like "if (STATE.error)" */
        return 0;

    bmap->size = size;
    bmap->width = grunt->width;
    bmap->bpl = bpl;
    bmap->height = grunt->height;
    bmap->lo_x = grunt->lo_x;
    bmap->hi_y = grunt->hi_y;
    bmap->i_dx = grunt->dx;
    bmap->i_dy = grunt->dy;

    /* OR the components into the final result */
    for (i = 0; i < num; i++)
    {
        FS_BITMAP *temp;

        temp = get_sbitmap(_PS_ ttf, grunt->xppm, grunt->yppm, comps[i].glyphCode, 1);
        if (STATE.error)
        {
            FSS_free_char(_PS_ bmap);
            return 0;
        }
        if (temp)
            merge_bitmap(_PS_ bmap, temp, comps[i].xOffset, comps[i].yOffset);
        FSS_free_char(_PS_ temp);
    }
    return bmap;
}

/****************************************************************/
static FS_BITMAP *merge_bitmap(_DS_ FS_BITMAP *dst, FS_BITMAP*src, FS_SHORT dx, FS_SHORT dy)
{
    FS_USHORT i, j, k;
    FS_BYTE *dp, *sp;

    /* displace the source bitmap */
    src->lo_x += dx;
    src->hi_y += dy;

    /* make sure src fits inside dst */
    if (src->lo_x < dst->lo_x || (src->lo_x + src->width) > (dst->lo_x + dst->width))
    {
        STATE.error = EBLC_COMP_SIZE;
        return dst;
    }
    if (src->hi_y > dst->hi_y || (src->hi_y - src->height) < (dst->hi_y - dst->height))
    {
        STATE.error = EBLC_COMP_SIZE;
        return dst;
    }

    /* offsets between bitmaps */
    dx = src->lo_x - dst->lo_x;
    dy = dst->hi_y - src->hi_y;

    /* starting row pointers in src and dst */
    sp = src->bits;
    dp = dst->bits + dy * dst->bpl;

    for (i = 0; i < src->height; i++)
    {
        /* j == column in src, k == column in dst */
        for (j = 0, k = dx + j; j < src->width; j++, k++)
        {
            if (sp[j >> 3] & fs_mask[j & 7])
                dp [k >> 3] |= fs_mask[k & 7];
        }
        dp += dst->bpl;
        sp += src->bpl;
    }
    return dst;
}

/******************************************************************/

#endif /* FS_EMBEDDED_BITMAP */
